// control.v
// Master timing controller for the LTC2385/2386/2387
// One or two lane, DDR mode
// 16 or 18 bits
// 5/10/15MHz output data rate, 5/10/15MHz input clock
// DJS 8/11/16
//
// All timing is controlled by a counter that is clocked by the fast clock clk.
// At certain counter values, the following things happen:
//		cnv_en <= 1			enable external FF to create the rising edge of CNV
//		cnv_en <= 0			cause the falling edge of CNV
//		clk_en <= 1			create pulse on clk_en to send CLK pulses to LTC2387
//		clk_en <= 0
//		data_load <=1	update data outputs
//		data_load <=0	falling edge used by DC890B to latch data

module control (
  input wire bits_18,
  input wire two_lane,
  input wire clk,		// high speed master clock
  input wire sync,	// set counter when sync is high
  output reg cnv_en,	// =0 resets the external FF used to retime CLK_IN to make CNV
  output reg clk_en,	// enables CLK pulses to ADC
  output reg data_load,	// signal to latch the data
  output reg cic_clk,	// signal to clock data through the boxcar filter
  output reg latch
);

// Local Parameters
parameter [5:0] start_count = 5'd23;	// Initial value for counter. Set when sync is high.
parameter [5:0] tcnvenl = 5'd21;  		// Time for cnv_en to go low; 4 high speed clock cycle after CLKIN rising edge
parameter [5:0] tcnvenh = 5'd4;			// Time for cnv_en to go high; 21 high speed clock cycle after CLKIN rising edge
parameter [5:0] tclkenh = 5'd3;			// Time for clk_en to go high; 22 high speed clock cycle after CLKIN rising edge

reg [5:0] tclkenl,tdataloadh,tdataloadl,tcicclkh,tcicclkl,tdsclkh,tdsclkl;
reg ds_clk;

always @(clk)
begin
  if(two_lane)
	begin
    if(bits_18)
		  tclkenl = 5'd22;		// Time for clk_en to go low; 3 high speed clock cycle after CLKIN rising edge
	 else
		  tclkenl = 5'd23;		// Time for clk_en to go low; 2 high speed clock cycle after CLKIN rising edge
	 tdataloadh = 5'd19;		// Time for data_load to go high; 6 high speed clock cycle after CLKIN rising edge
	 tdataloadl = 5'd16;		// Time for data_load to go low; 9 high speed clock cycle after CLKIN rising edge
	 tcicclkh = 5'd16;			// Time for cic_clk to go high; x high speed clock cycle after CLKIN rising edge
	 tcicclkl = 5'd14;			// Time for cic_clk to go low; x high speed clock cycle after CLKIN rising edge
	 tdsclkh = 5'd14;			// Time for ds_clk to go high; x high speed clock cycle after CLKIN rising edge
	 tdsclkl = 5'd12;			// Time for ds_clk to go low; x high speed clock cycle after CLKIN rising edge
	end
  else
	begin
	 if(bits_18)
		  tclkenl = 5'd18;		// Time for clk_en to go low; 7 high speed clock cycle after CLKIN rising edge  
	 else
		  tclkenl = 5'd19;		// Time for clk_en to go low; 6 high speed clock cycle after CLKIN rising edge
	 tdataloadh = 5'd15;		// Time for data_load to go high; 10 high speed clock cycle after CLKIN rising edge
	 tdataloadl = 5'd12;		// Time for data_load to go low; 13 high speed clock cycle after CLKIN rising edge
	 tcicclkh = 5'd12;			// Time for cic_clk to go high; x high speed clock cycle after CLKIN rising edge
	 tcicclkl = 5'd10;			// Time for cic_clk to go low; x high speed clock cycle after CLKIN rising edge
	 tdsclkh = 5'd10;			// Time for ds_clk to go high; x high speed clock cycle after CLKIN rising edge
	 tdsclkl = 5'd8;			// Time for ds_clk to go low; x high speed clock cycle after CLKIN rising edge
	end
end
  
reg [5:0] count;						// master counter
reg [7:0] ds_count;
initial ds_count = 8'd0;
parameter [7:0] ds_rate = 8'd128;

// Main
always @ (posedge clk)
    if (sync)
		  count <= start_count;
	 else 
		  count <= count - 1'b1;		
		
always @ (posedge clk)
	if (count == tcnvenh)
		cnv_en <= 1'b1;
	else if (count == tcnvenl)
		cnv_en <= 1'b0;
	else
		cnv_en <= cnv_en;		
				
always @ (posedge clk)
	if (count == tdataloadh)
		data_load <= 1'b1;
	else if (count == tdataloadl)
		data_load <= 1'b0;
	else
		data_load <= data_load;

always @ (posedge clk)
	if (count == tcicclkh)
		cic_clk <= 1'b1;
	else if (count == tcicclkl)
		cic_clk <= 1'b0;
	else
		cic_clk <= cic_clk;

always @ (posedge clk)
	if (count == tdsclkh)
		ds_clk <= 1'b1;
	else if (count == tdsclkl)
		ds_clk <= 1'b0;
	else
		ds_clk <= ds_clk;
		
always @ (posedge clk)
	if (count == tclkenh)
		clk_en <= 1'b1;
	else if (count == tclkenl)
		clk_en <= 1'b0;
	else
		clk_en <= clk_en;

//downsample data clock by 'ds_rate'
always @ (negedge ds_clk)
begin
	ds_count = ds_count + 8'd1;
	if(ds_count == ds_rate)
		ds_count = 8'd0;
	latch = (ds_count == (ds_rate - 8'd1))? 1'b0:1'b1;
end

endmodule